import { supabase, TABLES, ONBOARDING_STATUS } from '@/config/supabase';
import { Vendor, OnboardingRequest } from '@/types';

/**
 * Vendor Service - Handles vendor-related database operations
 */
export class VendorService {
  /**
   * Create a new vendor record
   */
  async createVendor(vendorData: {
    user_id: string;
    email: string;
    business_name: string | null;
    onboarding_status: string;
    subscription_plan: string;
  }) {
    try {
      const { data, error } = await supabase
        .from(TABLES.VENDORS)
        .insert([vendorData])
        .select()
        .single();

      return { data, error };
    } catch (error) {
      console.error('Create vendor error:', error);
      return { data: null, error };
    }
  }

  /**
   * Get vendor by user ID
   */
  async getVendorByUserId(userId: string) {
    try {
      const { data, error } = await supabase
        .from(TABLES.VENDORS)
        .select('*')
        .eq('user_id', userId)
        .single();

      return { data, error };
    } catch (error) {
      console.error('Get vendor by user ID error:', error);
      return { data: null, error };
    }
  }

  /**
   * Get vendor by ID
   */
  async getVendorById(vendorId: string) {
    try {
      const { data, error } = await supabase
        .from(TABLES.VENDORS)
        .select('*')
        .eq('id', vendorId)
        .single();

      return { data, error };
    } catch (error) {
      console.error('Get vendor by ID error:', error);
      return { data: null, error };
    }
  }

  /**
   * Update vendor onboarding information
   */
  async updateOnboarding(vendorId: string, onboardingData: OnboardingRequest) {
    try {
      const { data, error } = await supabase
        .from(TABLES.VENDORS)
        .update({
          business_name: onboardingData.business_name,
          profile_picture_url: onboardingData.profile_picture_url || null,
          onboarding_status: ONBOARDING_STATUS.COMPLETE,
          updated_at: new Date().toISOString()
        })
        .eq('id', vendorId)
        .select()
        .single();

      return { data, error };
    } catch (error) {
      console.error('Update onboarding error:', error);
      return { data: null, error };
    }
  }

  /**
   * Update vendor profile
   */
  async updateVendor(vendorId: string, updateData: Partial<Vendor>) {
    try {
      const { data, error } = await supabase
        .from(TABLES.VENDORS)
        .update({
          ...updateData,
          updated_at: new Date().toISOString()
        })
        .eq('id', vendorId)
        .select()
        .single();

      return { data, error };
    } catch (error) {
      console.error('Update vendor error:', error);
      return { data: null, error };
    }
  }

  /**
   * Get all vendors (admin only)
   */
  async getAllVendors() {
    try {
      const { data, error } = await supabase
        .from(TABLES.VENDORS)
        .select('*')
        .order('created_at', { ascending: false });

      return { data, error };
    } catch (error) {
      console.error('Get all vendors error:', error);
      return { data: null, error };
    }
  }

  /**
   * Delete vendor (admin only)
   */
  async deleteVendor(vendorId: string) {
    try {
      const { error } = await supabase
        .from(TABLES.VENDORS)
        .delete()
        .eq('id', vendorId);

      return { error };
    } catch (error) {
      console.error('Delete vendor error:', error);
      return { error };
    }
  }
}

export const vendorService = new VendorService(); 